UNIT GLObjects;

{ GLObjects   - Basic handling for scene object registration and definition of a
                few scene objects
  Version     - 0.1.13
  Last Change - 26 June 1997
  for more information see help file
}

INTERFACE

USES Controls, GLScene, Windows;

TYPE TCube = CLASS(TSceneObject)
     PROTECTED
       PROCEDURE BuildList; OVERRIDE;
     PUBLISHED
       PROPERTY Ambient;
       PROPERTY CSGGroup;
       PROPERTY Diffuse;
       PROPERTY RotationX;
       PROPERTY RotationY;
       PROPERTY RotationZ;
       PROPERTY Shininess;
       PROPERTY Specular;
       PROPERTY Texture; 
       PROPERTY X;
       PROPERTY Y;
       PROPERTY Z;
     END;

     TSphere = CLASS(TSceneObject)
     PROTECTED
       PROCEDURE BuildList; OVERRIDE;
     PUBLISHED
       PROPERTY Ambient;
       PROPERTY CSGGroup;
       PROPERTY Diffuse;
       PROPERTY Shininess;
       PROPERTY Specular;
       PROPERTY Texture; 
       PROPERTY X;
       PROPERTY Y;
       PROPERTY Z;
     END;

     TCylinder = CLASS(TSceneObject)
     PROTECTED
       PROCEDURE BuildList; OVERRIDE;
     PUBLISHED
       PROPERTY Ambient;
       PROPERTY CSGGroup;
       PROPERTY Diffuse;
       PROPERTY RotationX;
       PROPERTY RotationY;
       PROPERTY RotationZ;
       PROPERTY Shininess;
       PROPERTY Specular;
       PROPERTY Texture; 
       PROPERTY X;
       PROPERTY Y;
       PROPERTY Z;
     END;

     TCone = CLASS(TSceneObject)
     PROTECTED
       PROCEDURE BuildList; OVERRIDE;
     PUBLISHED
       PROPERTY Ambient;
       PROPERTY CSGGroup;
       PROPERTY Diffuse;
       PROPERTY RotationX;
       PROPERTY RotationY;
       PROPERTY RotationZ;
       PROPERTY Shininess;
       PROPERTY Specular;
       PROPERTY Texture; 
       PROPERTY X;
       PROPERTY Y;
       PROPERTY Z;
     END;

     TTorus = CLASS(TSceneObject)
     PROTECTED
       PROCEDURE BuildList; OVERRIDE;
     PUBLISHED
       PROPERTY Ambient;
       PROPERTY CSGGroup;
       PROPERTY Diffuse;
       PROPERTY RotationX;
       PROPERTY RotationY;
       PROPERTY RotationZ;
       PROPERTY Shininess;
       PROPERTY Specular;
       PROPERTY Texture; 
       PROPERTY X;
       PROPERTY Y;
       PROPERTY Z;
     END;

VAR ObjectStock : TComposite;       // a list of objects, which can be used for scene building
    ObjectIcons : TImageList;       // a list of icons for scene objects
    OverlayIndex,                   // indexes into the object icon list
    SceneRootIndex,
    CameraRootIndex,
    LightsourceRootIndex,
    ObjectRootIndex,
    StockObjectRootIndex : Integer;

FUNCTION GetClassFromIndex(Index: Integer): TSceneObjectClass;
FUNCTION GetImageIndex(ASceneObject: TSceneObjectClass) : Integer;

PROCEDURE GetRegisteredSceneObjects(VAR ObjectList: TStringList);
PROCEDURE RegisterSceneObject(ASceneObject: TSceneObjectClass; AName: STRING; AImage: HBitmap);
PROCEDURE UnRegisterSceneObject(ASceneObject: TSceneObjectClass; AName: STRING);

//------------------------------------------------------------------------------

IMPLEMENTATION

{$R GLObjects.RES}

USES GL, GLU, Graphics, SysUtils;

CONST UnitCube : ARRAY[0..7] OF TAffineFloatVector =
                 ((0,0,1),  // left  lower front
                  (0,1,1),  // right lower front
                  (1,1,1),  // right upper front
                  (1,0,1),  // left  upper front
                  (0,0,0),  // left  lower back
                  (0,1,0),  // left  upper back
                  (1,1,0),  // right upper back
                  (1,0,0)); // right lower back

TYPE PSceneObjectEntry = ^TSceneObjectEntry;
     {holds a relation between an scene object class, its global identification,
      its location in the object stock and its icon reference}
     TSceneObjectEntry = RECORD
                            ObjectClass : TSceneObjectClass;
                            Name : STRING[32];    // type name of the object
                            Index,                // index into "ObjectStock"
                            ImageIndex : Integer; // index into "ObjectIcons"
                          END;

VAR SceneObjectList : TList = NIL;

//----------------- predefined objects -----------------------------------------

PROCEDURE TCube.BuildList;

VAR T : THomogenIntVector;

BEGIN
  WITH BoundingBox DO
  BEGIN
    LeftLowerFront:=UnitCube[0];
    RightUpperBack:=UnitCube[6];
  END;

  // handle texture generation
  glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
  glEnable(GL_TEXTURE_GEN_S);
  glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
  glEnable(GL_TEXTURE_GEN_T);

  // draw the six faces of the cube with triangles
  // front
  T[0]:=1; T[1]:=0; T[2]:=0; T[3]:=0;
  glTexGeniv(GL_S,GL_OBJECT_PLANE,@T);
  T[0]:=0; T[1]:=1; T[2]:=0; T[3]:=0;
  glTexGeniv(GL_T,GL_OBJECT_PLANE,@T);
  glBegin(GL_TRIANGLE_STRIP);
    glNormal3f(0,0,1);
    glVertex3fv(@UnitCube[0]);
    glVertex3fv(@UnitCube[3]);
    glVertex3fv(@UnitCube[1]);
    glVertex3fv(@UnitCube[2]);
  glEnd;
  
  // bottom
  T[0]:=0; T[1]:=0; T[2]:=1; T[3]:=0;
  glTexGeniv(GL_T,GL_OBJECT_PLANE,@T);
  glBegin(GL_TRIANGLE_STRIP);
    glNormal3f(0,-1,0);
    glVertex3fv(@UnitCube[0]);
    glVertex3fv(@UnitCube[4]);
    glVertex3fv(@UnitCube[3]);
    glVertex3fv(@UnitCube[7]);
  glEnd;

  // back
  T[0]:=0; T[1]:=1; T[2]:=0; T[3]:=0;
  glTexGeniv(GL_T,GL_OBJECT_PLANE,@T);
  glBegin(GL_TRIANGLE_STRIP);
    glNormal3f(0,0,-1);
    glVertex3fv(@UnitCube[4]);
    glVertex3fv(@UnitCube[5]);
    glVertex3fv(@UnitCube[7]);
    glVertex3fv(@UnitCube[6]);
  glEnd;

  // top
  T[0]:=0; T[1]:=0; T[2]:=1; T[3]:=0;
  glTexGeniv(GL_T,GL_OBJECT_PLANE,@T);
  glBegin(GL_TRIANGLE_STRIP);
    glNormal3f(0,1,0);
    glVertex3fv(@UnitCube[1]);
    glVertex3fv(@UnitCube[2]);
    glVertex3fv(@UnitCube[5]);
    glVertex3fv(@UnitCube[6]);
  glEnd;

  // left
  T[0]:=0; T[1]:=1; T[2]:=0; T[3]:=0;
  glTexGeniv(GL_T,GL_OBJECT_PLANE,@T);
  T[0]:=0; T[1]:=0; T[2]:=1; T[3]:=0;
  glTexGeniv(GL_S,GL_OBJECT_PLANE,@T);
  glBegin(GL_TRIANGLE_STRIP);
    glNormal3f(-1,0,0);
    glVertex3fv(@UnitCube[0]);
    glVertex3fv(@UnitCube[1]);
    glVertex3fv(@UnitCube[4]);
    glVertex3fv(@UnitCube[5]);
  glEnd;

  // right
  glBegin(GL_TRIANGLE_STRIP);
    glNormal3f(1,0,0);
    glVertex3fv(@UnitCube[2]);
    glVertex3fv(@UnitCube[3]);
    glVertex3fv(@UnitCube[6]);
    glVertex3fv(@UnitCube[7]);
  glEnd;
END;

//------------------------------------------------------------------------------

PROCEDURE TSphere.BuildList;

VAR Sphere: PGLUQuadricObj;

BEGIN
  WITH BoundingBox DO
  BEGIN
    LeftLowerFront:=MakeAffineVector(-0.5,0.5,0.5);
    RightUpperBack:=MakeAffinevector(0.5,-0.5,-0.5);
  END;
  Sphere:=gluNewQuadric;
  gluQuadricNormals(Sphere,GLU_SMOOTH);
  gluQuadricDrawStyle(Sphere,GL_FILL);
  gluQuadricTexture(Sphere,GL_TRUE);
  gluSphere(Sphere,0.5,25,25);
  gluDeleteQuadric(Sphere);
END;

//------------------------------------------------------------------------------

PROCEDURE TCylinder.BuildList;

VAR QObject : PGLUQuadricObj;

BEGIN
  WITH BoundingBox DO
  BEGIN
    LeftLowerFront:=MakeAffineVector(-1,1,1);
    RightUpperBack:=MakeAffinevector(1,-1,-1);
  END;
  QObject:=gluNewQuadric;
  gluQuadricNormals(QObject,GLU_SMOOTH);
  gluQuadricDrawStyle(QObject,GL_FILL);
  gluQuadricOrientation(QObject,GLU_INSIDE);
  gluQuadricTexture(QObject,GL_TRUE);
  glTranslatef(0,0,-1);
  gluDisk(QObject,0,1,25,5);
  gluQuadricOrientation(QObject,GLU_OUTSIDE);
  gluCylinder(QObject,1,1,2,25,25);
  glTranslatef(0,0,2);
  gluDisk(QObject,0,1,25,5);
  gluDeleteQuadric(QObject);
END;

//------------------------------------------------------------------------------

PROCEDURE TCone.BuildList;

VAR QObject : PGLUQuadricObj;

BEGIN
  WITH BoundingBox DO
  BEGIN
    LeftLowerFront:=MakeAffineVector(-1,1,1);
    RightUpperBack:=MakeAffinevector(1,-1,-1);
  END;
  QObject:=gluNewQuadric;
  gluQuadricNormals(QObject,GLU_SMOOTH);
  gluQuadricDrawStyle(QObject,GL_FILL);
  gluQuadricOrientation(QObject,GLU_INSIDE);
  gluQuadricTexture(QObject,GL_TRUE);
  glTranslatef(0,0,-1);
  gluDisk(QObject,0,1,25,5);
  gluQuadricOrientation(QObject,GLU_OUTSIDE);
  gluCylinder(QObject,1,0,2,25,25);
  gluDeleteQuadric(QObject);
END;

//------------------------------------------------------------------------------

PROCEDURE TTorus.BuildList;

VAR I,J,
    Rings,Sides  : Integer;
    MinorRadius,
    MajorRadius,
    Theta,Phi,
    Theta1,Phi1  : GLdouble;
    DrawType     : GLEnum;
    p0,p1,
    p2,p3,
    n0,n1,
    n2,n3        : TAffineDblVector;

BEGIN
  // handle texture generation
  glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
  glEnable(GL_TEXTURE_GEN_S);
  glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
  glEnable(GL_TEXTURE_GEN_T);

  Rings:=25;
  Sides:=15;
  MinorRadius:=0.2;
  MajorRadius:=0.8;
  DrawType:=GL_QUADS; 
  FOR I:=0 TO Rings-1 DO
  BEGIN
    Theta:=(I*2*Pi)/Rings;
    Theta1:=((I+1)*2*Pi)/Rings;
    FOR J:=0 TO Sides-1 DO
    BEGIN
      Phi:=(J*2*Pi)/Sides;
      Phi1:=((J+1)*2*Pi)/Sides;

      p0[0]:=cos(theta)*(MajorRadius+MinorRadius*cos(phi));
      p0[1]:=-sin(theta)*(MajorRadius+MinorRadius*cos(phi));
      p0[2]:=MinorRadius*sin(phi);

      p1[0]:=cos(theta1)*(MajorRadius+MinorRadius*cos(phi));
      p1[1]:=-sin(theta1)*(MajorRadius+MinorRadius*cos(phi));
      p1[2]:=MinorRadius*sin(phi);

      p2[0]:=cos(theta1)*(MajorRadius+MinorRadius*cos(phi1));
      p2[1]:=-sin(theta1)*(MajorRadius+MinorRadius*cos(phi1));
      p2[2]:=MinorRadius*sin(phi1);

      p3[0]:=cos(theta)*(MajorRadius+MinorRadius*cos(phi1));
      p3[1]:=-sin(theta)*(MajorRadius+MinorRadius*cos(phi1));
      p3[2]:=MinorRadius*sin(phi1);

      n0[0]:=cos(theta)*(cos(phi));
      n0[1]:=-sin(theta)*(cos(phi));
      n0[2]:=sin(phi);

      n1[0]:=cos(theta1)*(cos(phi));
      n1[1]:=-sin(theta1)*(cos(phi));
      n1[2]:=sin(phi);

      n2[0]:=cos(theta1)*(cos(phi1));
      n2[1]:=-sin(theta1)*(cos(phi1));
      n2[2]:=sin(phi1);

      n3[0]:=cos(theta)*(cos(phi1));
      n3[1]:=-sin(theta)*(cos(phi1));
      n3[2]:=sin(phi1);

      glBegin(DrawType);
        glNormal3dv(@n3);
        glVertex3dv(@p3);
        glNormal3dv(@n2);
        glVertex3dv(@p2);
        glNormal3dv(@n1);
        glVertex3dv(@p1);
        glNormal3dv(@n0);
        glVertex3dv(@p0);
      glEnd();
    END;
  END;
END;

//----------------- end of predefined objects ----------------------------------

FUNCTION FindSceneObjectClass(AObjectClass: TSceneObjectClass; ASceneObject: STRING) : PSceneObjectEntry;

VAR I : Integer;

BEGIN
  Result:=NIL;
  I:=0;
  IF AObjectClass = NIL THEN AObjectClass:=TExternalObject;
  WITH SceneObjectList DO
  BEGIN
    FOR I:=0 TO Count-1 DO
      WITH TSceneObjectEntry(Items[I]^) DO
        IF (ObjectClass = AObjectClass)         AND
           (Length(ASceneObject) = 0)   OR
           (CompareText(Name,ASceneObject) = 0) THEN Break;
    IF I < Count THEN Result:=Items[I];
  END;
END;

//------------------------------------------------------------------------------

FUNCTION GetClassFromIndex(Index: Integer): TSceneObjectClass;

BEGIN
  IF Index < 0 THEN Index:=0;
  IF Index > SceneObjectList.Count-2 THEN Index:=SceneObjectList.Count-2;
  Result:=TSceneObjectEntry(SceneObjectList.Items[Index+2]^).ObjectClass;
END;

//------------------------------------------------------------------------------

FUNCTION GetImageIndex(ASceneObject: TSceneObjectClass) : Integer;

VAR AClassEntry : PSceneObjectEntry;
    AName       : STRING;

BEGIN
  IF ASceneObject = TExternalObject THEN AName:=TExternalObject(ASceneObject).TypeName
                                    ELSE AName:='';
  AClassEntry:=FindSceneObjectClass(ASceneObject,AName);
  IF assigned(AClassEntry) THEN Result:=AClassEntry^.ImageIndex;
END;

//------------------------------------------------------------------------------

PROCEDURE GetRegisteredSceneObjects(VAR ObjectList: TStringList);

VAR I : Integer;

BEGIN
  IF ObjectList = NIL THEN Exit;
  WITH ObjectList DO
  BEGIN
    Clear;
    // let light source and camera out
    FOR I:=2 TO SceneObjectList.Count-1 DO
      WITH TSceneObjectEntry(SceneObjectList.Items[I]^) DO Add(Name);
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE RegisterSceneObject(ASceneObject: TSceneObjectClass; AName: STRING; AImage: HBitmap);

// registers a stock object and adds it to the stock object list

VAR NewEntry  : PSceneObjectEntry;
    AInstance : TSceneObject;
    Pic       : TPicture;

BEGIN
  NewEntry:=NIL;
  IF ASceneObject = NIL THEN ASceneObject:=TExternalObject;
  WITH SceneObjectList DO
  BEGIN
    // make sure, no class is registered twice
    IF assigned(FindSceneObjectClass(ASceneObject,AName)) THEN Exit;
    GetMem(NewEntry,SizeOf(TSceneObjectEntry));
    IF NewEntry = NIL THEN RAISE Exception.Create('Could not allocate memory for scene object registration!');
    Pic:=TPicture.Create;
    TRY
      WITH NewEntry^ DO
      BEGIN
        // object stock stuff
        AInstance:=ASceneObject.Create(ObjectStock);
        AInstance.Name:=AName;
        ObjectStock.AddChild(AInstance);
        // registered objects list stuff
        ObjectClass:=ASceneObject;
        NewEntry^.Name:=AName;
        Index:=AInstance.Index;
        IF AImage <> 0 THEN
        BEGIN
          Pic.Bitmap.Handle:=AImage;
          ObjectIcons.AddMasked(Pic.Bitmap,Pic.Bitmap.Canvas.Pixels[0,0]);
          ImageIndex:=ObjectIcons.Count-1;
        END
        ELSE ImageIndex:=0;
      END;
      Add(NewEntry);
    FINALLY
      Pic.Free;
    END;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE UnRegisterSceneObject(ASceneObject: TSceneObjectClass; AName: STRING);

// unregisters a stock object and removes it from the stock object list

VAR OldEntry : PSceneObjectEntry;
    AObject  : TSceneObject;

BEGIN
  // find the class in the scene object list
  OldEntry:=FindSceneObjectClass(ASceneObject,AName);
  // found?
  IF assigned(OldEntry) THEN
  BEGIN
    // yes, so get its instance in "ObjectStock"
    AObject:=ObjectStock[OldEntry^.Index];
    // remove its entry from the list of registered objects
    SceneObjectList.Remove(OldEntry);
    // free the instance
    ObjectStock[OldEntry.Index].Free;
    // remove the instance entry from object stock
    ObjectStock.Remove(AObject,False);
    // finally free the memory for the entry in 
    FreeMem(OldEntry,SizeOf(TSceneObjectEntry))
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE CreateDefaultObjectIcons;

VAR Pic : TPicture;

BEGIN
  ObjectIcons:=TImageList.CreateSize(32,32);
  Pic:=TPicture.Create;
  WITH ObjectIcons, Pic.Bitmap.Canvas DO
  TRY
    {There's a more direct way for loading images into the image list, but
     the image quality suffers too much}
    Pic.Bitmap.Handle:=LoadBitmap(HInstance,'CROSS');
    AddMasked(Pic.Bitmap,Pixels[0,0]); OverlayIndex:=Count-1;
    Overlay(OverlayIndex,0); // used as indicator for disabled objects
    Pic.Bitmap.Handle:=LoadBitmap(HInstance,'UNIVERSE2');
    AddMasked(Pic.Bitmap,Pixels[0,0]); SceneRootIndex:=Count-1;
    Pic.Bitmap.Handle:=LoadBitmap(HInstance,'CAMERAS2');
    AddMasked(Pic.Bitmap,Pixels[0,0]); CameraRootIndex:=Count-1;
    Pic.Bitmap.Handle:=LoadBitmap(HInstance,'LAMPS2');
    AddMasked(Pic.Bitmap,Pixels[0,0]); LightsourceRootIndex:=Count-1;
    Pic.Bitmap.Handle:=LoadBitmap(HInstance,'OBJECTS2');
    AddMasked(Pic.Bitmap,Pixels[0,0]); ObjectRootIndex:=Count-1;
    AddMasked(Pic.Bitmap,Pixels[0,0]); StockObjectRootIndex:=Count-1;
  FINALLY
    Pic.Free;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE DestroySceneObjectList;

VAR I : Integer;

BEGIN
  WITH SceneObjectList DO
  BEGIN
    FOR I:=0 TO Count-1 DO FreeMem(Items[I],SizeOf(TSceneObjectEntry));
    Free;
  END;
END;

//------------------------------------------------------------------------------

INITIALIZATION
  SceneObjectList:=TList.Create;
  ObjectStock:=TComposite.Create(NIL);
  ObjectStock.Name:='ObjectStock';
  CreateDefaultObjectIcons;
  RegisterSceneObject(TLightSource,'Lightsource',LoadBitmap(HInstance,'LAMP'));
  RegisterSceneObject(TCamera,'Camera',LoadBitmap(HInstance,'CAMERA'));
  RegisterSceneObject(TComposite,'Composite',LoadBitmap(HInstance,'COMPOSITE'));
  RegisterSceneObject(TCube,'Cube',LoadBitmap(HInstance,'CUBE'));
  RegisterSceneObject(TSphere,'Sphere',LoadBitmap(HInstance,'SPHERE'));
  RegisterSceneObject(TCylinder,'Cylinder',LoadBitmap(HInstance,'CYLINDER'));
  RegisterSceneObject(TCone,'Cone',LoadBitmap(HInstance,'CONE'));
  RegisterSceneObject(TTorus,'Torus',LoadBitmap(HInstance,'TORUS'));
FINALIZATION
  DestroySceneObjectList;
  ObjectStock.Free;  // scene objects instances will be freed by the destructor
  ObjectIcons.Free;
END.
